<?php

use PhpOffice\PhpSpreadsheet\IOFactory;

/**
 * The admin-specific functionality of the plugin.
 *
 * @link       http://ays-pro.com/
 * @since      1.0.0
 *
 * @package    Ays_Pb
 * @subpackage Ays_Pb/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    Ays_Pb
 * @subpackage Ays_Pb/admin
 * @author     AYS Pro LLC <info@ays-pro.com>
 */
class Ays_Pb_Admin {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	private $popupbox_obj;
    private $settings_obj;
    private $popup_categories_obj;
    private $reports_obj;
    private $subscribes_obj;
    private $attributes_obj;
    private $each_submission_obj;
    private $capability;
    
	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string    $plugin_name       The name of this plugin.
	 * @param      string    $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version )
    {

        $this->plugin_name = $plugin_name;
        $this->version = $version;
        add_filter('set-screen-option', array(__CLASS__, 'set_screen'), 10, 3);
        $per_page_array = array(
            'popupboxes_per_page',
            'popup_categories_per_page',
            'pb_results_per_page',
            'pb_submissions_per_page',
        );
        foreach($per_page_array as $option_name){
            add_filter('set_screen_option_'.$option_name, array(__CLASS__, 'set_screen'), 10, 3);
        }

    }

	/**
	 * Register the stylesheets for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles($hook_suffix) {
        wp_enqueue_style($this->plugin_name . '-admin', plugin_dir_url(__FILE__) . 'css/admin.css', array(), $this->version, 'all');
		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Ays_Pb_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Ays_Pb_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

        // You need styling for the datepicker. For simplicity I've linked to the jQuery UI CSS on a CDN.
		wp_enqueue_style( $this->plugin_name . '-icon', plugin_dir_url( __FILE__ ) . 'css/ays-pb-icon.css', array(), $this->version, 'all' );
        if(false === strpos($hook_suffix, $this->plugin_name))
            return;
        
        // wp_register_style( 'jquery-ui', 'https://code.jquery.com/ui/1.12.1/themes/smoothness/jquery-ui.css' );
        // wp_enqueue_style( 'jquery-ui' );

		wp_enqueue_style( 'wp-color-picker' );
        // wp_enqueue_style( $this->plugin_name . '-font-awesome', AYS_PB_PUBLIC_URL . '/css/ays-pb-font-awesome.min.css', array(), $this->version, 'all' );
        wp_enqueue_style( $this->plugin_name . '-jquery-datetimepicker', plugin_dir_url(__FILE__) . 'css/jquery-ui-timepicker-addon.css', array(), $this->version, 'all');
        wp_enqueue_style( $this->plugin_name . '-bootstrap', plugin_dir_url(__FILE__) . 'css/bootstrap.min.css', array(), $this->version, 'all');
        wp_enqueue_style( $this->plugin_name . '-animate', AYS_PB_PUBLIC_URL . '/css/animate.css', array(), $this->version, 'all' );
        wp_enqueue_style( $this->plugin_name . '-select2', plugin_dir_url(__FILE__) . 'css/ays-pb-select2.min.css', array(), $this->version, 'all');
        wp_enqueue_style( $this->plugin_name . "-affiliate", plugin_dir_url( __FILE__ ) . 'css/ays-pb-affiliate.css', array(), time(), 'all' );
		wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/ays-pb-admin.css', array(), $this->version, 'all' );
        wp_enqueue_style( $this->plugin_name . '-dropdown', plugin_dir_url(__FILE__) .  'css/dropdown.min.css', array(), $this->version, 'all');
        wp_enqueue_style( $this->plugin_name . '-transition', plugin_dir_url(__FILE__) .  'css/transition.min.css', array(), $this->version, 'all');
	}

	/**
	 * Register the JavaScript for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts($hook_suffix) {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Ays_Pb_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Ays_Pb_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */
        $wp_post_types = get_post_types('', 'objects');
        $all_post_types = array();
        foreach ($wp_post_types as $pt){
            $all_post_types[] = array(
                $pt->name,
                $pt->label
            );
        }

        if (false !== strpos($hook_suffix, "plugins.php")){
            wp_enqueue_script( $this->plugin_name . '-sweetalert-js', plugin_dir_url( __FILE__ ) . 'js/ays-pb-sweetalert2.all.min.js', array('jquery'), $this->version, true );
            wp_enqueue_script( $this->plugin_name . '-admin', plugin_dir_url(__FILE__) . 'js/admin.js', array('jquery'), $this->version, true);
            wp_localize_script($this->plugin_name . '-admin', 'popup_box_ajax', array(
                'ajax_url' => admin_url('admin-ajax.php')
            ));
        }
        if(false === strpos($hook_suffix, $this->plugin_name))
            return;

        global $wp_version;
        if(Ays_Pb_Data::ays_version_compare($wp_version, '>=', '5.5' )){
            wp_enqueue_script( $this->plugin_name . '-wp-load-scripts', plugin_dir_url(__FILE__) . 'js/ays-wp-load-scripts.js', array(), $this->version, true);
        }

        wp_enqueue_script('jquery-ui-datepicker');
		wp_enqueue_script('jquery-effects-core');
        wp_enqueue_script('jquery-ui-sortable');
        wp_enqueue_media();

        wp_enqueue_script( $this->plugin_name . '-wp-color-picker-alpha', plugin_dir_url( __FILE__ ) . 'js/wp-color-picker-alpha.min.js',array( 'wp-color-picker' ),$this->version, true );

        $color_picker_strings = array(
            'clear'            => __( 'Clear', $this->plugin_name ),
            'clearAriaLabel'   => __( 'Clear color', $this->plugin_name ),
            'defaultString'    => __( 'Default', $this->plugin_name ),
            'defaultAriaLabel' => __( 'Select default color', $this->plugin_name ),
            'pick'             => __( 'Select Color', $this->plugin_name ),
            'defaultLabel'     => __( 'Color value', $this->plugin_name ),
        );
        wp_localize_script( $this->plugin_name . '-wp-color-picker-alpha', 'wpColorPickerL10n', $color_picker_strings );

        wp_enqueue_script(  $this->plugin_name . '-functions', plugin_dir_url(__FILE__) . 'js/partials/functions.js', array('jquery', 'wp-color-picker'), $this->version, true);
        wp_localize_script( $this->plugin_name . '-functions', 'aysPbFunctionLangaugeData', array(
            'AYS_PB_ADMIN_URL'      => AYS_PB_ADMIN_URL,
            'AYS_PB_PUBLIC_URL'     => AYS_PB_PUBLIC_URL,

            'editImage'             => __( "Edit Image", $this->plugin_name ),
        ));

        wp_enqueue_script( $this->plugin_name . '-sweetalert-js', plugin_dir_url( __FILE__ ) . 'js/ays-pb-sweetalert2.all.min.js', array('jquery'), $this->version, true );
        wp_enqueue_script( $this->plugin_name . '-popper', plugin_dir_url( __FILE__ ) . 'js/popper.min.js', array( 'jquery' ), $this->version, true );
        wp_enqueue_script( $this->plugin_name . '-bootstrap', plugin_dir_url( __FILE__ ) . 'js/bootstrap.min.js', array( 'jquery' ), $this->version, true );
        wp_enqueue_script( $this->plugin_name . '-select2-js', plugin_dir_url( __FILE__ ) . 'js/ays-pb-select2.min.js', array('jquery'), $this->version, true );
        wp_enqueue_script( $this->plugin_name . '-custom-dropdown-adapter', plugin_dir_url( __FILE__ ) . 'js/ays-select2-dropdown-adapter.js', array('jquery'), $this->version, true );
        wp_enqueue_script( $this->plugin_name . '-jquery.datetimepicker.js', plugin_dir_url( __FILE__ ) . 'js/jquery-ui-timepicker-addon.js', array( 'jquery' ), $this->version, true );

        // JS export Files
        wp_enqueue_script( $this->plugin_name."-xlsx.core.min.js", plugin_dir_url( __FILE__ ) . 'js/xlsx.core.min.js', array( 'jquery' ), $this->version, true );
        wp_enqueue_script( $this->plugin_name."-fileSaver.js", plugin_dir_url( __FILE__ ) . 'js/FileSaver.js', array( 'jquery' ), $this->version, true );
        wp_enqueue_script( $this->plugin_name."-jhxlsx.js", plugin_dir_url( __FILE__ ) . 'js/jhxlsx.js', array( 'jquery' ), $this->version, true );

        /* 
        =======================================================
           Popup admin dashboard script (Statistics)
        =======================================================
        */
        if ( strpos($hook_suffix, 'statistics') !== false ) {
            wp_enqueue_script( $this->plugin_name . '-charts-google', plugin_dir_url(__FILE__) . 'js/google-chart.js', array('jquery'), $this->version, true);
            wp_enqueue_script( $this->plugin_name . '-statistics', plugin_dir_url(__FILE__) . 'js/partials/ays-pb-admin-global-statistics-charts.js', array('jquery'), $this->version, true);
        }

		wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/ays-pb-admin.js', array( 'jquery', 'wp-color-picker' ), $this->version, false );
        wp_localize_script($this->plugin_name, 'pb', array(
            'ajax'                  => admin_url('admin-ajax.php'),
            'post_types'            => $all_post_types,
            'AYS_PB_ADMIN_URL'      => AYS_PB_ADMIN_URL,
            'AYS_PB_PUBLIC_URL'     => AYS_PB_PUBLIC_URL,

            'editImage'             => __( "Edit Image", $this->plugin_name ),
            'addImage'              => __( "Add Image", $this->plugin_name ),
            'pleaseSubscribe'       => __( "Please subscribe to receive it in your inbox.", $this->plugin_name ),

            'customContent'         => __( "Custom Content" , $this->plugin_name ),
            'shortcode'             => __( "Shortcode" , $this->plugin_name ),
            'subscribe'             => __( "Subscribe" , $this->plugin_name ),
            'subscription'          => __( "Subscription" , $this->plugin_name ),
            'yesOrNo'               => __( "Yes or No" , $this->plugin_name ),
            'video'                 => __( "Video" , $this->plugin_name ),
            'iframe'                => __( "Iframe" , $this->plugin_name ),
            'contactForm'           => __( "Contact form" , $this->plugin_name ),
            'subscribeAndGetFile'   => __( "Subscribe and get a file" , $this->plugin_name ),
            'coupon'                => __( "Coupon" , $this->plugin_name ),
            'countdown'             => __( "Countdown" , $this->plugin_name ),
            'acceptCookie'          => __( "Accept cookie" , $this->plugin_name ),
            'cookiePopup'           => __( "Cookie popup" , $this->plugin_name ),
            'woo'                   => __( "WooCommerce Product" , $this->plugin_name ),

            'yes'                   => __( "Yes" , $this->plugin_name ),
            'no'                    => __( "No" , $this->plugin_name ),
            'sendToAdmin'           => __( "Send to admin" , $this->plugin_name ),
            'download'              => __( "Download" , $this->plugin_name ),
            'logIn'                 => __( "Log In" , $this->plugin_name ),
            'googleMap'             => __( "Google Map" , $this->plugin_name ),

            'frozenInTheEnd'        => __( "In the End", $this->plugin_name ),
            'frozenMostOften'       => __( "Most often, northern icebergs are 1-2 km long, but there are also those that reach 200 km.", $this->plugin_name ),
            'frozenLearnMore'       => __( "Learn More", $this->plugin_name ),

            'foodKeepUpdated'       => __( "Keep Updated", $this->plugin_name ),
            'foodSpammyFresh'       => __( "We have spammy fresh seafood, and our chef cooks according to the most delicious recipes", $this->plugin_name ),
            'foodLearnMore'         => __( "Learn More", $this->plugin_name ),
            
            'bookDescHeader'        => __( "Stay updated", $this->plugin_name ),
            'bookDescPara'          => __( "Over 2 billion people follow newsletters, are you one of them?", $this->plugin_name ),
            'bookContent'           => __( "Get to know more about what's going on around the glob.", $this->plugin_name ),
            'bookContentButton'     => __( "Find out", $this->plugin_name ),

            'holidayHeader'         => __( "Holiday Deal", $this->plugin_name ),
            'holidayContent'        => __( "Holidays are near, Join us to find the best holiday spots.", $this->plugin_name ),
            'holidayButton'         => __( "Join Now", $this->plugin_name ),

            'forestExclusiveOffer'  => __( "Get Exclusive offers and Discounts", $this->plugin_name ),
            'forestTakeOff'         => __( "Take 10% off Your Next Purchase", $this->plugin_name ),
            'forestSubscribe'       => __( "Subscribe To our newsletters and get 10% off coupon with free", $this->plugin_name ),
            'forestGetStarted'      => __( "Get Started", $this->plugin_name ),

            'loadResource'          => __( "Can't load resource.", $this->plugin_name ),
            'somethingWentWrong'    => __( "Maybe something went wrong.", $this->plugin_name ),
            'pleaseEnterMore'       => __( "Please select more", $this->plugin_name ),
        ));

        wp_enqueue_script( $this->plugin_name . '-admin-ajax', plugin_dir_url( __FILE__ ) . 'js/ays-pb-admin-ajax.js', array( 'jquery'), $this->version, true );

        wp_enqueue_script( $this->plugin_name . '-dropdown-min', plugin_dir_url(__FILE__) . 'js/dropdown.min.js', array('jquery'), $this->version, true);
        wp_enqueue_script( $this->plugin_name . '-transition-min', plugin_dir_url(__FILE__) . 'js/transition.min.js', array('jquery'), $this->version, true);

	}

    /**
	 * De-register JavaScript files for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function disable_scripts($hook_suffix) {
        if (false !== strpos($hook_suffix, $this->plugin_name)) {
            if (is_plugin_active('ai-engine/ai-engine.php')) {
                wp_deregister_script('mwai');
                wp_deregister_script('mwai-vendor');
                wp_dequeue_script('mwai');
                wp_dequeue_script('mwai-vendor');
            }
        }
	}

    // Code Mirror
    function codemirror_enqueue_scripts($hook) {
        if (false === strpos($hook, $this->plugin_name)){
            return;
        }
        if(function_exists('wp_enqueue_code_editor')){
            $cm_settings['codeEditor'] = wp_enqueue_code_editor(array(
                'type' => 'text/css',
                'codemirror' => array(
                    'inputStyle' => 'contenteditable',
                    'theme' => 'cobalt',
                   
                )
            ));

            wp_enqueue_script('wp-theme-plugin-editor');
            wp_localize_script('wp-theme-plugin-editor', 'cm_settings', $cm_settings);
        
            wp_enqueue_style('wp-codemirror');
        }
    }

    /**
     * Register the administration menu for this plugin into the WordPress Dashboard menu.
     *
     * @since    1.0.0
     */
    public function add_plugin_admin_menu() {
        
        $this->capability = $this->popup_box_capabilities();
        $capability = $this->popup_box_capabilities();
        
        $hook_popupbox = add_menu_page(
            __('Popup Box'),
            __('Popup Box'),
            $this->capability,
            $this->plugin_name,
            array($this, 'display_plugin_setup_page'),
            plugin_dir_url(__FILE__) . '/assets/icons/popup.svg',
            6);
    }
        
    public function add_plugin_popups_submenu() {
        
        $hook_popupbox = add_submenu_page(
            $this->plugin_name,
            __('Popups', $this->plugin_name),
            __('Popups', $this->plugin_name),
            $this->capability,
            $this->plugin_name,
            array($this, 'display_plugin_setup_page')
        );
        
        add_action( "load-$hook_popupbox", array( $this, 'screen_option_popupbox' ) );
        add_action( "load-$hook_popupbox", array( $this, 'add_tabs' ));
    }

    public function add_plugin_categories_submenu() {

        $hook_categories = add_submenu_page(
            $this->plugin_name,
            __('Categories', $this->plugin_name),
            __('Categories', $this->plugin_name),
            $this->capability,
            $this->plugin_name . '-categories',
            array($this, 'display_plugin_categories_page')
        );

        add_action( "load-$hook_categories", array( $this, 'screen_option_categories' ) );
        add_action( "load-$hook_categories", array( $this, 'add_tabs' ));
    }

    public function add_plugin_custom_fields_submenu(){
        $hook_popup_attributes = add_submenu_page(
            $this->plugin_name,
            __('Custom Fields', $this->plugin_name),
            __('Custom Fields', $this->plugin_name),
            $this->capability,
            $this->plugin_name . '-attributes',
            array($this, 'display_plugin_attributes_page')
        );

        add_action("load-$hook_popup_attributes", array($this, 'screen_option_attributes'));
        add_action("load-$hook_popup_attributes", array( $this, 'add_tabs' ));
    }

    public function add_plugin_reports_submenu(){
        $results_text = __('Analytics', $this->plugin_name);
        $hook_reports = add_submenu_page(
            $this->plugin_name,
            $results_text,
            $results_text,
            $this->capability,
            $this->plugin_name . '-reports',
            array($this, 'display_plugin_results_page')
        );

        add_action("load-$hook_reports", array($this, 'screen_option_reports'));
        add_action("load-$hook_reports", array( $this, 'add_tabs' ));

        $hook_statistics = add_submenu_page(
            'statistics_slug',
            __('Analytics', $this->plugin_name),
            null,
            $this->capability,
            $this->plugin_name . '-statistics',
            array($this, 'display_plugin_statistics_page')
        );

        add_action("load-$hook_statistics", array($this, 'screen_option_statistics'));
        add_action("load-$hook_statistics", array( $this, 'add_tabs' ));

        add_filter('parent_file', array($this,'ays_pb_select_submenu'));
    }

    public function add_plugin_submissions_submenu(){
        global $wpdb;
        $sql = "SELECT COUNT(*) FROM {$wpdb->prefix}ays_pb_subscribes WHERE `read` = 0";
        $unread_results_count = $wpdb->get_var($sql);
        $results_text = __('Submissions', $this->plugin_name); 
        $menu_item = ($unread_results_count == 0) ? $results_text : $results_text . '<span class="ays_menu_badge ays_results_bage">' . $unread_results_count . '</span>';
        $hook_submissions = add_submenu_page(
            $this->plugin_name,
            $results_text,
            $menu_item,
            $this->capability,
            $this->plugin_name . '-submissions',
            array($this, 'display_plugin_submissions_page')
        );

        add_action("load-$hook_submissions", array($this, 'screen_option_submissions'));
        add_action( "load-$hook_submissions", array( $this, 'add_tabs' ));

        $hook_each_submission = add_submenu_page(
            'each_submission_slug',
            __('Each', $this->plugin_name),
            null,
            $this->capability,
            $this->plugin_name . '-each-submission',
            array($this, 'display_plugin_each_submission_page')
        );

        add_action("load-$hook_each_submission", array($this, 'screen_option_each_popup_submission'));
        add_filter('parent_file', array($this,'popup_box_select_submenu'));
    }

    public function add_plugin_settings_submenu(){
        $hook_settings = add_submenu_page( $this->plugin_name,
            __('General Settings', $this->plugin_name),
            __('General Settings', $this->plugin_name),
            'manage_options',
            $this->plugin_name . '-settings',
            array($this, 'display_plugin_settings_page') 
        );
        add_action("load-$hook_settings", array($this, 'screen_option_settings'));
        add_action( "load-$hook_settings", array( $this, 'add_tabs' ));
    }
    
    public function add_plugin_export_import_submenu(){
        $hook_export_import = add_submenu_page(
            $this->plugin_name,
            __('Export/Import', $this->plugin_name),
            __('Export/Import', $this->plugin_name),
            $this->capability,
            $this->plugin_name . '-export_import',
            array($this, 'display_plugin_export_import_page')
        );
        add_action( "load-$hook_export_import", array( $this, 'add_tabs' ));
    }

    public function add_plugin_affiliate_submenu(){
        $hook_affiliates = add_submenu_page( $this->plugin_name,
            __('Affiliates', $this->plugin_name),
            __('Affiliates', $this->plugin_name),
            $this->capability,
            $this->plugin_name . '-affiliates',
            array($this, 'display_plugin_affiliate_page') 
        );
        add_action( "load-$hook_affiliates", array( $this, 'add_tabs' ));
    }

    public function add_plugin_featured_plugins_submenu(){
        $hook_reports = add_submenu_page( $this->plugin_name,
            __('Our Products', $this->plugin_name),
            __('Our Products', $this->plugin_name),
            $this->capability,
            $this->plugin_name . '-featured-plugins',
            array($this, 'display_plugin_featured_plugins_page') 
        );
        add_action( "load-$hook_reports", array( $this, 'add_tabs' ));
    }

    public function ays_pb_select_submenu($file) {
        global $plugin_page;
        if ( $this->plugin_name . "-statistics" == $plugin_page ) {
            $plugin_page = $this->plugin_name."-reports";
        }

        return $file;
    }

    public function admin_menu_styles(){

        echo "<style>
            .ays_menu_badge{
                color: #fff;
                display: inline-block;
                font-size: 10px;
                line-height: 14px;
                text-align: center;
                background: #ca4a1f;
                margin-left: 5px;
                border-radius: 20px;
                padding: 2px 5px;
            }

            #adminmenu a.toplevel_page_ays-pb div.wp-menu-image img {
                width: 23px;
                padding: 3px 0 0;
                transition: .3s ease-in-out;
            }
        </style>";

    }

    protected function popup_box_capabilities(){
        global $wpdb;
        $sql = "SELECT meta_value FROM {$wpdb->prefix}ays_pb_settings WHERE `meta_key` = 'user_roles'";
        $result = $wpdb->get_var($sql);
        
        $capability = 'manage_options';
        if($result !== null){
            $ays_user_roles = json_decode($result, true);
            if(is_user_logged_in()){
                $current_user = wp_get_current_user();
                $current_user_roles = $current_user->roles;
                $ishmar = 0;
                foreach($current_user_roles as $r){
                    if(in_array($r, $ays_user_roles)){
                        $ishmar++;
                    }
                }
                if($ishmar > 0){
                    $capability = "read";
                }
            }
        }
        return $capability;
    }

    /**
     * Add settings action link to the plugins page.
     *
     * @since    1.0.0
     */
    public function add_action_links( $links ) {
        /*
        *  Documentation : https://codex.wordpress.org/Plugin_API/Filter_Reference/plugin_action_links_(plugin_file_name)
        */
        $settings_link = array(
            '<a href="' . admin_url( 'options-general.php?page=' . $this->plugin_name ) . '">' . __('Settings', $this->plugin_name) . '</a>',
            '<a href="https://ays-demo.com/popup-box-plugin-pro-demo/" target="_blank">' . __('Demo', $this->plugin_name) . '</a>',
        );
        return array_merge(  $settings_link, $links );

    }

    /**
     * Render the settings page for this plugin.
     *
     * @since    1.0.0
     */
    public function display_plugin_setup_page() {
		$action = (isset($_GET['action'])) ? sanitize_text_field( $_GET['action'] ) : '';

        include_once 'partials/actions/ays-pb-admin-countries.php';
		
        switch ( $action ) {
            case 'add':
                include_once( 'partials/actions/ays-pb-admin-actions.php' );
                break;
            case 'edit':
                include_once( 'partials/actions/ays-pb-admin-actions.php' );
                break;
            default:
                include_once( 'partials/ays-pb-admin-display.php' );
        }
    }

    public function display_plugin_categories_page(){
        $action = (isset($_GET['action'])) ? sanitize_text_field($_GET['action']) : '';

        switch ($action) {
            case 'add':
                include_once('partials/actions/ays-pb-categories-actions.php');
                break;
            case 'edit':
                include_once('partials/actions/ays-pb-categories-actions.php');
                break;
            default:
                include_once('partials/ays-pb-categories-display.php');
        }
    }

    public function display_plugin_attributes_page(){
        $action = (isset($_GET['action'])) ? sanitize_text_field($_GET['action']) : '';

        switch ($action) {
            case 'add':
                include_once('partials/attributes/actions/ays-pb-attributes-actions.php');
                break;
            case 'edit':
                include_once('partials/attributes/actions/ays-pb-attributes-actions.php');
                break;
            default:
                include_once('partials/attributes/ays-pb-attributes-display.php');
        }
    }

    public function display_plugin_results_page(){
        include_once('partials/reports/ays-pb-reports-display.php');
    }

    public function display_plugin_statistics_page(){
        include_once 'partials/reports/ays-pb-reports-statistics-display.php';
    }

    public function display_plugin_submissions_page(){
        include_once('partials/submissions/ays-pb-submissions-display.php');
    }

    public function display_plugin_each_submission_page(){
        include_once 'partials/submissions/ays-pb-each-submission-display.php';
    }

    public function display_plugin_settings_page(){
        include_once('partials/settings/popup-box-settings.php');
    }
    
	public function display_plugin_export_import_page(){
        include_once('partials/export-import/popup-box-export-import.php');
    }

    public function display_plugin_affiliate_page(){
        include_once('partials/affiliate/popup-box-affiliate-display.php');
    }

    public function display_plugin_featured_plugins_page(){
        include_once('partials/features/ays-pb-plugin-featured-display.php');
    }

	public static function set_screen( $status, $option, $value ) {
        return $value;
    }
	
    public static function ays_pb_restriction_string($type, $x, $length){
        $output = "";
        switch($type){
            case "char":                
                if(strlen($x)<=$length){
                    $output = $x;
                } else {
                    $output = substr($x,0,$length) . '...';
                }
                break;
            case "word":
                $res = explode(" ", $x);
                if(count($res)<=$length){
                    $output = implode(" ",$res);
                } else {
                    $res = array_slice($res,0,$length);
                    $output = implode(" ",$res) . '...';
                }
            break;
        }
        return $output;
    }

	public function screen_option_popupbox() {
		$option = 'per_page';
		$args   = array(
			'label'   => __('PopupBox', $this->plugin_name),
			'default' => 20,
			'option'  => 'popupboxes_per_page'
		);

        if( ! ( isset( $_GET['action'] ) && ( $_GET['action'] == 'add' || $_GET['action'] == 'edit' ) ) ){
            add_screen_option( $option, $args );
        }

		$this->popupbox_obj = new Ays_PopupBox_List_Table($this->plugin_name);
        $this->settings_obj = new Ays_PopupBox_Settings_Actions($this->plugin_name);
	}

    public function screen_option_statistics(){
        $this->reports_obj = new Ays_PopupBox_Reports_List_Table( $this->plugin_name );
    }

    public function screen_option_settings() {
        $this->settings_obj = new Ays_PopupBox_Settings_Actions($this->plugin_name);
    }

    public function screen_option_categories() {
        $option = 'per_page';
        $args   = array(
            'label'   => __('Categories', $this->plugin_name),
            'default' => 20,
            'option'  => 'popup_categories_per_page'
        );

        if( ! ( isset( $_GET['action'] ) && ( $_GET['action'] == 'add' || $_GET['action'] == 'edit' ) ) ){
            add_screen_option($option, $args);
        }

        $this->popup_categories_obj = new Popup_Categories_List_Table($this->plugin_name);
        $this->settings_obj = new Ays_PopupBox_Settings_Actions($this->plugin_name);
    }

    public function screen_option_attributes(){
        $option = 'per_page';
        $args = array(
            'label' => __('Attributes', $this->plugin_name),
            'default' => 20,
            'option' => 'popup_attributes_per_page'
        );

        if( ! ( isset( $_GET['action'] ) && ( $_GET['action'] == 'add' || $_GET['action'] == 'edit' ) ) ){
            add_screen_option($option, $args);
        }

        $this->attributes_obj = new Ays_PopupBox_Attributes_List_Table($this->plugin_name);
    }

    public function screen_option_reports(){
        $option = 'per_page';
        $args = array(
            'label' => __('Reports', $this->plugin_name),
            'default' => 50,
            'option' => 'pb_results_per_page'
        );

        add_screen_option($option, $args);
        $this->reports_obj = new Ays_PopupBox_Reports_List_Table($this->plugin_name);
    }

    public function screen_option_submissions(){
        $option = 'per_page';
        $args = array(
            'label' => __('Submissions', $this->plugin_name),
            'default' => 50,
            'option' => 'pb_submissions_per_page'
        );

        add_screen_option($option, $args);
        $this->subscribes_obj = new Ays_PopupBox_Submissions_notices_List_Table($this->plugin_name);
    }

    public function screen_option_each_popup_submission() {
        $option = 'per_page';
        $args = array(
            'label' => __('Results', $this->plugin_name),
            'default' => 50,
            'option' => 'pb_each_submission_per_page',
        );

        add_screen_option($option, $args);
        $this->each_submission_obj = new Ays_PopupBox_Each_Submissions_List_Table($this->plugin_name);        
        $this->settings_obj = new Ays_PopupBox_Settings_Actions($this->plugin_name);
    }

    public function ays_popup_admin_ajax(){
        global $wpdb;

        $response = array(
            "status" => false
        );
        $function = isset($_REQUEST['function']) ? sanitize_text_field( $_REQUEST['function'] ) : null;

        if($function !== null){
            $response = array();        
            if( is_callable( array( $this, $function ) ) ){
                $response = $this->$function();

                ob_end_clean();
                $ob_get_clean = ob_get_clean();
                echo json_encode( $response );
                wp_die();
            }
        }
        ob_end_clean();
        $ob_get_clean = ob_get_clean();
        echo json_encode( $response );
        wp_die();
    }

    public function deactivate_plugin_option() {
        error_reporting(0);
        $request_value = isset( $_REQUEST['upgrade_plugin'] ) ? sanitize_text_field( $_REQUEST['upgrade_plugin'] ) : '';
    
        $upgrade_option = get_option( 'ays_pb_upgrade_plugin', '' );
    
        if ( $upgrade_option === '' ) {
            add_option( 'ays_pb_upgrade_plugin', $request_value );
        } else {
            update_option( 'ays_pb_upgrade_plugin', $request_value );
        }
        $response = array(
            'option' => get_option( 'ays_pb_upgrade_plugin', '' ),
        );
    
        wp_send_json_success( $response );
    }

    public function get_selected_options_pb() {

        if (isset($_POST['data']) && !empty($_POST['data'])) {
            $posts = get_posts(array(
                'post_type'   => $_POST['data'],
                'post_status' => 'publish',
                'numberposts' => -1

            ));
        } else {
            $posts = array();
        }

        $arr = array();
        foreach ( $posts as $post ) {
            array_push($arr, array($post->ID, $post->post_title));

        }

        echo json_encode($arr);
        wp_die();
    }

    public function ays_pb_add_reports() {

        global $wpdb;
        $pb_reports_table = $wpdb->prefix . "ays_pb_reports";
        $id = ( $_POST["id"] != NULL ) ? absint( intval( $_POST["id"] ) ) : null;

        if ($id !== null) {

            $start_date         = $_POST["startDate"];
            $end_date           = $_POST["endDate"];
            $duration           = $_POST["duration"];
            $count              = 1;
            $read               = 1;

            $result = $wpdb->insert(
                $pb_reports_table,
                array(
                    "pb_id"         => $id,
                    "count"         => $count,
                    "start_date"    => $start_date,
                    "end_date"      => $end_date,
                    "duration"      => $duration,
                    "read"          => $read
                ),
                array( "%d","%s","%s","%s","%s","%d" )
            );
            echo json_encode($result);
        }
        wp_die();
    }

    public function ays_pb_export_json(){
        global $wpdb;
        error_reporting(0);
        
        $ays_pb_table = $wpdb->prefix .'ays_pb';
        $ays_pb_cat_table = $wpdb->prefix .'ays_pb_categories';
       
        if (isset($_REQUEST['action']) && $_REQUEST['action'] == 'ays_pb_export_json') {
            $pb_ids = isset($_REQUEST['ays_pb_export']) ? $_REQUEST['ays_pb_export'] : array();
            
            $sql = "SELECT pb.*, pbcat.`title` AS cat_title, pbcat.`published` AS cat_status
                    FROM {$ays_pb_table} AS pb 
                    JOIN {$ays_pb_cat_table} AS pbcat 
                    ON pb.`category_id` = pbcat.`id`";
            
            if(empty($pb_ids)){
                $sql .= '';
            }else{
                $sql .= ' WHERE pb.`id` IN ('. implode(',', $pb_ids) .') ';
            }

            $results = $wpdb->get_results($sql, 'ARRAY_A');
            
            $data = array();
            $data['ays_pb_key'] = 1;
            
            foreach($results as $key => &$pb_data){
                $data['popup_boxes'][] = $pb_data;
            }

            $response = array(
                'status' => true,
                'data'   => $data,
                'title'  => 'popup-box-export',
            );
            
            echo json_encode($response);
            wp_die();
        }
    }

    public function ays_pb_import_json($import_file){
        global $wpdb;
        $name_arr = explode('.', $import_file['name']);
        $type     = end($name_arr);
        
        $json      = file_get_contents($import_file['tmp_name']);
        $json      = json_decode($json, true);
        $json_key  = isset($json['ays_pb_key']) ? $json['ays_pb_key'] : false;
        
        $failed_popup = 0;
        if($json_key){
            $ays_pb_table = $wpdb->prefix .'ays_pb';
            $ays_pb_category_table = $wpdb->prefix .'ays_pb_categories';
            $popup_boxes = $json['popup_boxes'];

            $imported_popup = 0;
            foreach($popup_boxes as $key => $value){
                $wpdb->insert(
                    $ays_pb_category_table,
                    array( 
                        'title'      =>  $value['cat_title'],
                        'published'  =>  $value['cat_status'],
                    ),
                    array( '%s' )
                );
                $pb_category = $wpdb->insert_id;
                    
                $category_id = $pb_category;

                $result = $wpdb->insert(
                        $ays_pb_table,         
                        array(         
                            'title'                => $value['title'],
                            'popup_name'           => $value['popup_name'],
                            'description'          => $value['description'],
                            'category_id'          => $category_id,
                            'autoclose'            => $value['autoclose'],
                            'cookie'               => $value['cookie'],
                            'width'                => $value['width'],
                            'height'               => $value['height'],
                            'bgcolor'              => $value['bgcolor'],
                            'textcolor'            => $value['textcolor'],
                            'bordersize'           => $value['bordersize'],
                            'bordercolor'          => $value['bordercolor'],
                            'border_radius'        => $value['border_radius'],
                            'shortcode'            => $value['shortcode'],
                            'users_role'           => $value['users_role'],
                            'custom_class'         => $value['custom_class'],
                            'custom_css'           => $value['custom_css'],
                            'custom_html'          => $value['custom_html'],
                            'onoffswitch'          => $value['onoffswitch'],
                            'show_all'             => $value['show_all'],
                            'delay'                => $value['delay'],
                            'scroll_top'           => $value['scroll_top'],
                            'animate_in'           => $value['animate_in'],
                            'animate_out'          => $value['animate_out'],
                            'action_button'        => $value['action_button'],
                            'view_place'           => $value['view_place'],
                            'action_button_type'   => $value['action_button_type'],
                            'modal_content'        => $value['modal_content'],
                            'view_type'            => $value['view_type'],
                            'onoffoverlay'         => $value['onoffoverlay'],
                            'show_popup_title'     => $value['show_popup_title'],
                            'show_popup_desc'      => $value['show_popup_desc'],
                            'close_button'         => $value['close_button'],
                            'header_bgcolor'       => $value['header_bgcolor'],
                            'bg_image'             => $value['bg_image'],
                            'log_user'             => $value['log_user'],
                            'guest'                => $value['guest'],
                            'active_date_check'    => $value['active_date_check'],
                            'activeInterval'       => $value['activeInterval'],
                            'deactiveInterval'     => $value['deactiveInterval'],
                            'options'              => $value['options'],
                            'pb_position'          => $value['pb_position'],
                            'pb_margin'            => $value['pb_margin'],
                        ),
                        array(
                            '%s', // title
                            '%s', // popup_name
                            '%s', // description
                            '%d', // category_id
                            '%d', // autoclose
                            '%d', // cookie
                            '%d', // width
                            '%d', // height
                            '%s', // bgcolor
                            '%s', // textcolor
                            '%d', // bordersize
                            '%s', // bordercolor
                            '%d', // border_radius
                            '%s', // shortcode
                            '%s', // users_role
                            '%s', // custom_class
                            '%s', // custom_css
                            '%s', // custom_html
                            '%s', // onoffswitch
                            '%s', // show_all
                            '%d', // delay
                            '%d', // scroll_top
                            '%s', // animate_in
                            '%s', // animate_out
                            '%s', // action_button
                            '%s', // view_place
                            '%s', // action_button_type
                            '%s', // modal_content
                            '%s', // view_type
                            '%s', // onoffoverlay
                            '%s', // show_popup_title
                            '%s', // show_popup_desc
                            '%s', // close_button
                            '%s', // header_bgcolor
                            '%s', // bg_image
                            '%s', // log_user
                            '%s', // guest
                            '%s', // active_date_check
                            '%s', // activeInterval
                            '%s', // deactiveInterval
                            '%s', // options
                            '%s', // pb_position
                            '%d'  // pb_margin
                    )
                );
                $imported_popup++;
            }
            $status = array(
                'popup_successed' => $imported_popup,
                'popup_failed' => '',
            );
            return $status;
        }
        $status = array(
            'popup_successed' => false,
            'popup_failed' => $failed_popup,
        );
        return $status;
    }

    public function add_tabs() {
		$screen = get_current_screen();

		if ( ! $screen) {
			return;
		}

		$screen->add_help_tab(
			array(
				'id'      => 'popupbox_help_tab',
				'title'   => __( 'General Information:
                    '),
				'content' =>
					'<h2>' . __( 'Popup Information', $this->plugin_name) . '</h2>' .
					'<p>' .
						__( 'The WordPress Popup plugin will help you to create engaging popups with fully customizable and responsive designs. Attract your audience and convert them into email subscribers or paying customers.  Construct advertising offers, generate more leads by creating option forms and subscription popups.',  $this->plugin_name ).'</p>'
			)
		);

		$screen->set_help_sidebar(
			'<p><strong>' . __( 'For more information:', $this->plugin_name) . '</strong></p>' .
			'<p>
                <a href=" https://www.youtube.com/watch?v=YSf6-icT2Ro&list=PL18_gEiPDg8Ocrbwn1SUjs2XaSZlgHpWj" target="_blank">' . __( 'Youtube video tutorials', $this->plugin_name ) . '</a>
            </p>' .
			'<p>
                <a href="https://ays-pro.com/wordpress-popup-box-plugin-user-manual" target="_blank">' . __( 'Documentation: ', $this->plugin_name ) . '</a>
            </p>' .
			'<p>
                <a href="https://ays-pro.com/wordpress/popup-box" target="_blank">' . __( 'Popup Box plugin Premium version:', $this->plugin_name ) . '</a>
            </p>'
		);
	}

    // Generate Coupons
    public function generate_coupons(){

        if ($_FILES['coupon_data']['size'] == 0 && $_FILES['coupon_data']['error'] == 0){
            ob_end_clean();
            $ob_get_clean = ob_get_clean();
            $response_text = __( "Please choose the file", $this->plugin_name );
            echo json_encode(array(
                'status' => false,
                'message' => $response_text,
            ));
            wp_die();
        }
        if(isset($_REQUEST['action']) && $_REQUEST['action'] == 'generate_coupons'){
            
            $name_arr = explode('.', $_FILES['coupon_data']['name']);
            $type     = end($name_arr);
            
            if($type == 'csv'){

                require_once(AYS_PB_DIR . 'includes/PHPExcel/vendor/autoload.php');
                $spreadsheet = IOFactory::load($_FILES['coupon_data']['tmp_name']);
                $coupon_sheet_names = $spreadsheet->getSheetNames();
                $pb_coupons_data = array();
                $pb_coupons = array();

                foreach ($coupon_sheet_names as $coupon_sheet_names_key => $coupon_sheet_name) {

                    $current_sheet = $spreadsheet->getSheet($coupon_sheet_names_key);
                    $highest_row = $current_sheet->getHighestRow();
                    $highest_column = $current_sheet->getHighestColumn();

                    $pb_coupons[$coupon_sheet_names_key] = $coupon_sheet_name;

                    for ($row = 1; $row <= $highest_row; $row++){

                        //  Read a row of data into an array
                        $ready_array = $current_sheet->rangeToArray('A' . $row . ':' . $highest_column . $row, "", false, true );

                        //  Insert row data array into your database of choice here
                        $ready_array = array_values( $ready_array );
                        $pb_coupons_data[$coupon_sheet_names_key][] = $ready_array[0];

                    }
                }

                $ready_data = array();
                foreach ($pb_coupons_data as $pb_coupon_data_key => $pb_coupon_data) {

                    foreach($pb_coupon_data as $coupon_data_key => $pb_coupon_data_value){

                        $coupons_array = array();

                        foreach($pb_coupon_data_value as $s_key => $s_value){

                            if($s_value != ''){
                                $coupons_array[] = $s_value;
                            }
                        }

                        $ready_data[] = $coupons_array;
                    }
                }

                ob_end_clean();
                $ob_get_clean = ob_get_clean();
                echo json_encode(array(
                    'status' => true,
                    'coupons_ready_data' => $ready_data,
                ));
                wp_die();
            }else{
                ob_end_clean();
                $ob_get_clean = ob_get_clean();
                $response_text = __( "File type should be a 'CSV'", $this->plugin_name );
                echo json_encode(array(
                    'status' => false,
                    'message' => $response_text,
                ));
                wp_die();
            }
        }else{
            ob_end_clean();
            $ob_get_clean = ob_get_clean();
            $response_text = __( "Something went wrong", $this->plugin_name );
            echo json_encode(array(
                'status' => false,
                'message' => $response_text,
            ));
            wp_die();
        }
    }

    public function ays_pb_update_database_tables () {
        if(isset($_REQUEST['action']) && $_REQUEST['action'] == 'ays_pb_update_database_tables'){
            $ays_pb_db_version = '1.0.0';
            update_site_option( 'ays_pb_db_version', $ays_pb_db_version );
            
            ob_end_clean();
            $ob_get_clean = ob_get_clean();
            echo json_encode(array(
                'status' => true,
            ));
            wp_die();
        }else{
            ob_end_clean();
            $ob_get_clean = ob_get_clean();
            $response_text = __( "Something went wrong", $this->plugin_name );
            echo json_encode(array(
                'status' => false,
                'message' => $response_text,
            ));
            wp_die();
        }
    }

    public static function get_listtables_title_length( $listtable_name ) {
        global $wpdb;

        $settings_table = $wpdb->prefix . "ays_pb_settings";
        $sql = "SELECT meta_value FROM ".$settings_table." WHERE meta_key = 'options'";
        $result = $wpdb->get_var($sql);
        $options = ($result == "") ? array() : json_decode(stripcslashes($result), true);
        $listtable_title_length = 5;
        if(! empty($options) ){
            switch ( $listtable_name ) {
                case 'popups':
                    $listtable_title_length = (isset($options['popup_title_length']) && intval($options['popup_title_length']) != 0) ? absint(intval($options['popup_title_length'])) : 5;
                    break; 
                case 'categories':
                    $listtable_title_length = (isset($options['categories_title_length']) && intval($options['categories_title_length']) != 0) ? absint(intval($options['categories_title_length'])) : 5;
                    break;
                default:
                    $listtable_title_length = 5;
                    break;
            }
            return $listtable_title_length;
        }
        return $listtable_title_length;
    }

    public function popup_box_admin_footer($a){
        if(isset($_REQUEST['page'])){
            if(false !== strpos($_REQUEST['page'], $this->plugin_name)){
                ?>
                <p style="font-size:13px;text-align:center;font-style:italic;">
                    <span style="margin-left:0px;margin-right:10px;" class="ays_heart_beat"><img src="<?php echo AYS_PB_ADMIN_URL . "/images/icons/hearth.svg"?>"></span>
                    <span><?php echo __( "If you love our plugin, please do big favor and rate us on WordPress.org", $this->plugin_name); ?></span> 
                    <a target="_blank" class="ays-rated-link" href='http://bit.ly/3kYanHL'>
                        <span class="ays-dashicons ays-dashicons-star-empty"></span>
                    	<span class="ays-dashicons ays-dashicons-star-empty"></span>
                    	<span class="ays-dashicons ays-dashicons-star-empty"></span>
                    	<span class="ays-dashicons ays-dashicons-star-empty"></span>
                    	<span class="ays-dashicons ays-dashicons-star-empty"></span>
                    </a>
                    <span class="ays_heart_beat"><img src="<?php echo AYS_PB_ADMIN_URL . "/images/icons/hearth.svg"?>"></span>
                </p>
            <?php
            }
        }
    }

    public function get_current_popup_statistic(){
        $popup_id = (isset($_REQUEST['popup_id']) && $_REQUEST['popup_id'] != "") ? absint( $_REQUEST['popup_id'] ) : 0;
        $data = array(
            'count' => array(),
            'avg'   => array()
        );

        $data['count'] = Ays_PopupBox_Reports_List_Table::get_results_dates( $popup_id );
        $data['avg'] = Ays_PopupBox_Reports_List_Table::get_dates_avg_duration( $popup_id );

        $result = array(
            'count' => array(
                'status'    => false,
                'message'   => __( 'There is no data to show.', $this->plugin_name )
            ),
            'avg'   => array(
                'status'    => false,
                'message'   => __( 'There is no data to show.', $this->plugin_name )
            )
        );

        if( ! empty(  $data['count'] ) ){
            $result['count'] = array(
                'status'        => true,
                'datesCounts'   =>  $data['count']
            );
        }

        if( ! empty(  $data['avg'] ) ){
            $result['avg'] = array(
                'status'    => true,
                'datesAvgs' =>  $data['avg']
            );
        }

        return $result;
    }

    public function get_next_or_prev_row_by_id( $id, $type = "next", $table = "ays_pb" ) {
        global $wpdb;
    
        if ( is_null( $table ) || empty( $table ) ) {
            return null;
        }
    
        $ays_table = esc_sql( $wpdb->prefix . $table );
    
        $where = array();
        $where_condition = "";
    
        $id     = (isset( $id ) && $id != "" && absint($id) != 0) ? absint( sanitize_text_field( $id ) ) : null;
        $type   = (isset( $type ) && $type != "") ? sanitize_text_field( $type ) : "next";
    
        if ( is_null( $id ) || $id == 0 ) {
            return null;
        }
    
        switch ( $type ) {
            case 'prev':
                $where[] = ' `id` < ' . $id . ' ORDER BY `id` DESC ';
            break;
            case 'next':
            default:
                $where[] = ' `id` > ' . $id;
                break;
        }
    
        if( ! empty($where) ){
            $where_condition = " WHERE " . implode( " AND ", $where );
        }
    
        $sql = "SELECT `id` FROM {$ays_table} ". $where_condition ." LIMIT 1";
        $results = $wpdb->get_row( $sql, 'ARRAY_A' );
    
        return $results;
    
    }

    public function ays_pb_show_results(){
        global $wpdb;
        $results_table = $wpdb->prefix . "ays_pb_subscribes";

        if (isset($_REQUEST['action']) && $_REQUEST['action'] == 'ays_pb_show_results') {
            $id = absint(intval($_REQUEST['result']));
            $results = Ays_Pb_Data::get_subscribe_by_id($id);
            $attibutes = Ays_Pb_Data::get_all_attributes();

            $user_id = intval($results['user_id']);
            $pb_id = intval($results['pb_id']);
            
            $user = get_user_by('id', $user_id);
            
            $user_ip = $results['user_ip'];
            $options = json_decode($results['options'] , true);
            $user_attributes = ( isset($options['popup_attributes']) ) ? $options['popup_attributes'] : array();
            $subscribe_date = $results['date'];

            
            $json = json_decode(file_get_contents("http://ipinfo.io/{$user_ip}/json"));
            $country = isset($json->country) ? $json->country : '';
            $region = isset($json->region) ? $json->region : '';
            $city = isset($json->city) ? $json->city : '';
            $from = $city . ', ' . $region . ', ' . $country . ', ' . $user_ip;

            $row = "<table id='ays-results-table'>";
            $row .= '<tr class="ays_result_element">
                        <td>'.__('Subscribe Date',$this->plugin_name).'</td>
                        <td colspan="3">' . $subscribe_date . '</td>
                    </tr>';
            if ($user_ip != '') {
                $row .= '<tr class="ays_result_element">
                            <td>'.__('User',$this->plugin_name).' IP</td>
                            <td colspan="3">' . $from . '</td>
                        </tr>';
            }
            
            $user_name = $user_id === 0 ? __( "Guest", $this->plugin_name ) : $user->data->display_name;
            if($user_id !== 0){
                $row .= '<tr class="ays_result_element">
                        <td>'.__('User',$this->plugin_name).' ID</td>
                        <td colspan="3">' . $user_id . '</td>
                    </tr>';
            }
            $row .= '<tr class="ays_result_element">
                    <td>'.__('User',$this->plugin_name).'</td>
                    <td colspan="3">' . $user_name . '</td>
                </tr>';                

            
            if(isset($user_attributes['ays_pb_form_name']) && $user_attributes['ays_pb_form_name'] !== ''){
                $row .= "<tr class=\"ays_result_element\">
                            <td>".__('Name',$this->plugin_name)."</td>
                            <td colspan='3'>".stripslashes($user_attributes['ays_pb_form_name'])."</td>
                        </tr>";
            }
            if(isset($results['subscribe_email']) && $results['subscribe_email'] !== ''){
                $row .= "<tr class=\"ays_result_element\">
                        <td>".__('Email',$this->plugin_name)."</td>
                        <td colspan='3'>".stripslashes($results['subscribe_email'])."</td>
                     </tr>";
            }
            if(isset($user_attributes['ays_pb_form_message']) && $user_attributes['ays_pb_form_message'] !== ''){
                $row .= "<tr class=\"ays_result_element\">
                        <td>".__('Message',$this->plugin_name)."</td>
                        <td colspan='3'>".stripslashes($user_attributes['ays_pb_form_message'])."</td>
                     </tr>";
            }
            
            if(isset($user_attributes['ays_pb_form_name'])){
                unset($user_attributes['ays_pb_form_name']);
            }
            if(isset($user_attributes['ays_pb_form_email'])){
                unset($user_attributes['ays_pb_form_email']);
            }
            if(isset($user_attributes['ays_pb_form_message'])){
                unset($user_attributes['ays_pb_form_message']);
            }

            if (isset($user_attributes)) {
            
                foreach ($user_attributes as $name => $value) {
                    $attr_name = '';
                    if(isset($attibutes[$name])){
                        $attr_name = isset($attibutes[$name]['name']) && $attibutes[$name]['name'] != '' ? esc_attr($attibutes[$name]['name']) : '';
                    }
                    if(stripslashes($value) == ''){
                        $attr_value = '-';
                    }else{
                        $attr_value = stripslashes($value);
                    }
                    
                    if($attr_value == 'on'){
                        $attr_value = __('Checked',$this->plugin_name);
                    }
                    if($attr_name){
                        $row .= '<tr class="ays_result_element">
                                <td>' . stripslashes($attr_name) . '</td>
                                <td colspan="3">' . $attr_value . '</td>
                            </tr>';
                    }
                }
            }

            $row .= "</table>";

            $subscribes_table = $results_table;
            $res           = $wpdb->update(
                $subscribes_table,
                array('read' => 1),
                array('read' => 0),
                array('%d'),
                array('%d')
            );

            
            // $sql = "UPDATE $results_table SET `read`=1 WHERE `id`=$id";
            // $wpdb->get_var($sql);
            ob_end_clean();
            $ob_get_clean = ob_get_clean();
            echo json_encode(array(
                "status" => true,
                "rows" => $row
            ));
            wp_die();
        }
    }

    public function popup_box_select_submenu($file) {
        global $plugin_page;
        if ($this->plugin_name."-each-submission" == $plugin_page) {
            $plugin_page = $this->plugin_name."-submissions";
        }
        return $file;
    }

    public static function ays_pb_get_woocommerce_products(){
        global $wpdb;

        $results = array();

        $sql = "SELECT t.*
                FROM {$wpdb->prefix}posts AS t
                WHERE t.post_type IN ('product')
                    AND t.post_status = 'publish' ORDER BY t.post_title ASC";

        $products = $wpdb->get_results( $sql );

        foreach ($products as $key => $value) {
            $results[] = array(
                'id' => $value->ID,
                'text' => $value->post_title,
            );
        }
    
        return $results;
    }

    public function ays_pb_results_export_file($path){
        global $wpdb;
        $popup_id = isset($_REQUEST['popup_id']) && $_REQUEST['popup_id'] != '' ? intval($_REQUEST['popup_id']) : 0;
        $sql = "SELECT *
                FROM {$wpdb->prefix}ays_pb_subscribes WHERE pb_id = '".$popup_id."'
                ORDER BY id DESC";

        $results = $wpdb->get_results($sql);
        $export_data = $this->ays_results_export_xlsx($results);

        ob_end_clean();
        $ob_get_clean = ob_get_clean();
        echo json_encode($export_data);
        wp_die();
    }


    public function ays_results_export_xlsx($results){
        
        global $wpdb;
        error_reporting(0);
        
        $results_array = array();
        $results_headers = array(
            array( 'text' => "wp_user" ),
            array( 'text' => "user_ip" ),
            array( 'text' => "subscribe_name" ),
            array( 'text' => "subscribe_email" ),
            array( 'text' => "popup_id" ),
            array( 'text' => "date" ),
        );        
        foreach ($attributes as $attribute){
            $results_headers[] = array( 'text' => $attribute->name );
        }
        $results_array[] = $results_headers;
        
        foreach ($results as $key => $result){
            $result = (array)$result;
            $wp_user = $result['user_id'] > 0 ? get_user_by('ID', $result['user_id'])->display_name : __("Guest", $this->plugin_name);
            $res_array = array(
                array( 'text' => $wp_user ),
                array( 'text' => $result['user_ip'] ),
                array( 'text' => $result['user_name'] ),
                array( 'text' => $result['subscribe_email'] ),
                array( 'text' => $result['pb_id'] ),
                array( 'text' => $result['date'] ),                
            );
            $results_array[] = $res_array;
        }
        
        $response = array(
            'status' => true,
            'data'   => $results_array,
            "type"   => 'xlsx'
        );
        return $response;
    }

    public function close_warning_note_permanently() {
        $cookie_expiration = time() + 60*60*24*30;
        setcookie('ays_pb_show_warning_note', 'ays_pb_show_warning_note_value', $cookie_expiration, '/');
    }

    public function ays_pb_create_author() {
        error_reporting(0);

        $search = isset($_REQUEST['search']) && $_REQUEST['search'] != '' ? $_REQUEST['search'] : null;
        $checked = isset($_REQUEST['val']) && $_REQUEST['val'] !='' ? $_REQUEST['val'] : null;
        $args = array();
        $arg = '';

        if($search !== null){
            $arg .= $search;
            $arg .= '*';
            $args['search'] = $arg;
        }
        
        $users = get_users( $args ); 
        $response = array(
            'results' => array()
        );

        if(empty($args)){
            $reports_users = '';
        }

        foreach ($users as $key => $user) {
            if ($checked !== null) {
                if ($user->ID == $checked) {
                    continue;
                }else{
                    $response['results'][] = array(
                        'id' => $user->ID,
                        'text' => $user->display_name
                    );
                }
            }else{
                $response['results'][] = array(
                    'id' => $user->ID,
                    'text' => $user->display_name,
                );
            }
        }     

        ob_end_clean();
        echo json_encode($response);
        wp_die();
    }
}
